import pygame
import math

WIDTH, HEIGHT = 800, 600
HALF_HEIGHT = HEIGHT // 2

FOV = math.pi / 3
HALF_FOV = FOV / 2
NUM_RAYS = 120
MAX_DEPTH = 800
DELTA_ANGLE = FOV / NUM_RAYS
DIST = NUM_RAYS / (2 * math.tan(HALF_FOV))
PROJ_COEFF = 3 * DIST * 50
SCALE = WIDTH // NUM_RAYS

# Visual settings
render_distance = 0.00005
Mainbrightness = 200


class Raycaster:
    def __init__(self, world_map):
        self.world = world_map

    # -------------------------------
    # WALL RAYCASTING
    # -------------------------------
    def cast_rays(self, screen, player):
        cur_angle = player.angle - HALF_FOV
        px, py = player.pos

        for ray in range(NUM_RAYS):
            sin_a = math.sin(cur_angle)
            cos_a = math.cos(cur_angle)

            for depth in range(MAX_DEPTH):
                x = px + depth * 0.01 * cos_a
                y = py + depth * 0.01 * sin_a

                if (int(x), int(y)) in self.world.map:
                    wall_type = self.world.map[(int(x), int(y))]
                    base_color = self.world.wall_colors[wall_type]

                    depth *= math.cos(player.angle - cur_angle)  # fisheye correction
                    proj_height = PROJ_COEFF / (depth + 0.0001)

                    brightness = Mainbrightness / (1 + depth * depth * render_distance)
                    brightness = max(30, min(255, brightness))

                    r = int(base_color[0] * (brightness / 255))
                    g = int(base_color[1] * (brightness / 255))
                    b = int(base_color[2] * (brightness / 255))

                    pygame.draw.rect(
                        screen,
                        (r, g, b),
                        (ray * SCALE, HALF_HEIGHT - proj_height // 2, SCALE, proj_height)
                    )
                    break
            cur_angle += DELTA_ANGLE

    # -------------------------------
    # SPRITES (ENEMIES + PROJECTILES)
    # -------------------------------
    def draw_sprite(self, screen, player, sprite_x, sprite_y, color, scale=1.0):
        dx = sprite_x - player.x
        dy = sprite_y - player.y
        distance = math.hypot(dx, dy)

        angle = math.atan2(dy, dx) - player.angle
        angle = (angle + math.tau) % math.tau
        if angle > math.pi:
            angle -= math.tau

        if abs(angle) > FOV:
            return

        if not self.is_visible(player, sprite_x, sprite_y):
            return

        screen_x = (angle / FOV + 0.5) * WIDTH
        size = min(700 / (distance + 0.0001), 400)
        radius = int(size // 2 * scale)
        center_y = HALF_HEIGHT

        pygame.draw.circle(screen, color, (int(screen_x), int(center_y)), radius)



    def render_sprites(self, screen, player, enemies, projectiles=[]):
        # enemies
        for e in enemies:
            if e.alive:
                self.draw_sprite(screen, player, e.x, e.y, e.color, scale=1.0)

        # projectiles
        for p in projectiles:
            if p.alive:
                self.draw_sprite(screen, player, p.x, p.y, p.color, scale=0.1)  # smaller


    # -------------------------------
    # LINE OF SIGHT CHECK
    # -------------------------------
    def is_visible(self, player, target_x, target_y, step=0.02):
        """Return True if no wall blocks the view from player to target."""
        dx = target_x - player.x
        dy = target_y - player.y
        dist = math.hypot(dx, dy)
        steps = int(dist / step)
        for i in range(1, steps + 1):
            ix = player.x + dx * i / steps
            iy = player.y + dy * i / steps
            if (int(ix), int(iy)) in self.world.map:
                return False
        return True

